#!/usr/bin/env python3
import os, subprocess, tempfile, sys, time

AFTER_EFFECTS_APP = os.environ.get("AE_APP_NAME", "Adobe After Effects 2025")

# --- Paths (edit if needed) ---
PROJECT     = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Baseball-Animation-3.aep"
CSV_PATH    = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
COMMON_JSX  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/gl_common.jsxinc"

# This file name must match the JSX file below
JSX_BATCH   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_HomeRun_3.jsx"

# AE / template
COMP        = "HomeRun-384"  # can override via env
RS_TEMPLATE = "Best Settings"
OM_TEMPLATE = "PNG Sequence"

# HomeRunText directory (files: HomeRunText-1.png .. -4.png)
HRTEXT_DIR  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/BaseBall/HomeRunText"

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/HomeRun_Animation_3-384"
PATH_TEMPLATE = "{league}"  # GL.outPaths() will add /{abbr}/{anim}/00000.png

# League (CLI override optional)
LEAGUE = "MLB"

# Controls
AE_DEBUG          = "1"
PURGE_BEFORE      = "1"
NO_RENDER         = "0"
QUIT_APP          = "0"    # keep AE open across runs
TEXT_STROKE_WIDTH = "0.5"

# Logos
LOGO_DIR  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large/"
LOGO_TPL  = "{league}/{abbr}"
LOGO_EXTS = "png,jpg,jpeg,svg,ai,psd"

# Layer names (override via env if your comp uses different names)
LAYER_LOGO_PRECOMP          = "LogoPreComp"       # precomp on root
SUBLAYER_LOGO_IN_PRECOMP    = "Logo"              # inside LogoPreComp
LAYER_LOGO_FLAT             = "TeamLogo"          # optional plain logo layer on root (no effect)
LAYER_HRTEXT                = "HomeRunText"       # footage layer to swap per run
LAYER_WATERFALL_PRECOMP     = "HomeRunWaterfall"  # precomp containing many tinted layers

# Existing text/shape colorizing bits
LAYER_HOMERUN    = "HOMERUN"                      # main HR text on root
LAYER_HR_PRECOMP = "RunScored PreComp"            # inner with HOME/RUN sublayers
INNER_LAYERS     = "HOME,RUN"

def _write_bridge_jsx(env):
    def esc(s): return str(s).replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(v)+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def _osascript(script_text):
    proc = subprocess.run(["osascript", "-s", "o"], input=script_text, text=True, capture_output=True)
    out = (proc.stdout or "").strip()
    err = (proc.stderr or "").strip()
    if err:
        print("\n[osascript stderr]\n" + err, file=sys.stderr)
    proc.check_returncode()
    return out, err

def _ping_ae(app_name):
    try:
        _osascript(f'tell application "{app_name}" to return name')
        return True
    except subprocess.CalledProcessError:
        return False

def run_after_effects(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa_src = f'''
set bridgePosix to "{bridge}"
set bridgeFile to POSIX file bridgePosix
with timeout of 86400 seconds
  tell application "{AFTER_EFFECTS_APP}"
    activate
    try
      set res to DoScriptFile bridgeFile
      return "AERUN:OK:" & (res as text)
    on error errMsg number errNum
      return "AERUN:ERR:" & errNum & ":" & errMsg
    end try
  end tell
end timeout
'''
    _osascript(osa_src)

def make_env(anim_name, runnum):
    return {
        # core
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,
        "AE_COMMON_JSX": COMMON_JSX,

        # naming / colorization
        "AE_LAYER_HOMERUN": LAYER_HOMERUN,
        "AE_LAYER_HR_PRECOMP": LAYER_HR_PRECOMP,
        "AE_INNER_STROKE_NAMES": INNER_LAYERS,
        "AE_TEXT_STROKE_WIDTH": TEXT_STROKE_WIDTH,

        # logo targets
        "AE_LAYER_LOGO_PRECOMP": LAYER_LOGO_PRECOMP,
        "AE_SUBLAYER_LOGO_IN_PRECOMP": SUBLAYER_LOGO_IN_PRECOMP,
        "AE_LAYER_LOGO2": LAYER_LOGO_FLAT,  # optional (no-op if missing)

        # logos location
        "AE_LOGO_DIR": LOGO_DIR,
        "AE_LOGO_PATH_TEMPLATE": LOGO_TPL,
        "AE_LOGO_EXTS": LOGO_EXTS,

        # HomeRunText swap
        "AE_LAYER_HRTEXT": LAYER_HRTEXT,
        "AE_HRTEXT_DIR": HRTEXT_DIR,
        "AE_RUNNUM": str(runnum),

        # Waterfall tinting
        "AE_LAYER_WATERFALL_PRECOMP": LAYER_WATERFALL_PRECOMP,

        # output & templates
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": anim_name,
        "AE_RS_TEMPLATE": RS_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        # controls
        "AE_PURGE_BEFORE_RENDER": PURGE_BEFORE,
        "AE_NO_RENDER": NO_RENDER,
        "AE_QUIT": QUIT_APP,
        "AE_LEAGUE": LEAGUE,
        "AE_DEBUG": AE_DEBUG,
    }

if __name__ == "__main__":
    print("=== After Effects Home Run Batch ===")
    if not _ping_ae(AFTER_EFFECTS_APP):
        print(f'!! Could not talk to "{AFTER_EFFECTS_APP}"', file=sys.stderr)
        sys.exit(2)

    # RunNum 1..4
    for runnum in range(1, 5):
        anim_name = f"H-{runnum}"
        print(f"\n== RunNum {runnum} :: AE_ANIM = {anim_name}")
        run_after_effects(make_env(anim_name, runnum))
        time.sleep(2)  # small breather for AE
    print("\nDone.")
